<?php

// Registro del tipo de contenido personalizado para CSS Snippets
add_action('init', 'register_css_snippet_post_type');
function register_css_snippet_post_type() {
    register_post_type('css_snippet', [
        'labels' => [
            'name' => __('CSS Snippets', 'code-snippet-manager'),
            'singular_name' => __('CSS Snippet', 'code-snippet-manager'),
            'add_new' => __('Add New CSS Snippet', 'code-snippet-manager'),
            'add_new_item' => __('Add New CSS Snippet', 'code-snippet-manager'),
            'edit_item' => __('Edit CSS Snippet', 'code-snippet-manager'),
            'view_item' => __('View CSS Snippet', 'code-snippet-manager'),
        ],
        'public' => false, 
        'show_ui' => true, 
        'show_in_menu' => false, // Deshabilita el menú separado para CSS
        'supports' => ['title', 'revisions'], 
        'menu_icon' => 'dashicons-editor-css', 
    ]);
}

// Registro del tipo de contenido personalizado para JS Snippets
add_action('init', 'register_js_snippet_post_type');
function register_js_snippet_post_type() {
    register_post_type('js_snippet', [
        'labels' => [
            'name' => __('JavaScript Snippets', 'code-snippet-manager'),
            'singular_name' => __('JavaScript Snippet', 'code-snippet-manager'),
            'add_new' => __('Add New JS Snippet', 'code-snippet-manager'),
            'add_new_item' => __('Add New JS Snippet', 'code-snippet-manager'),
            'edit_item' => __('Edit JS Snippet', 'code-snippet-manager'),
            'view_item' => __('View JS Snippet', 'code-snippet-manager'),
        ],
        'public' => false, 
        'show_ui' => true, 
        'show_in_menu' => false, // Deshabilita el menú separado para JS
        'supports' => ['title', 'revisions'], 
        'menu_icon' => 'dashicons-editor-javascript', 
    ]);
}


add_action('admin_menu', 'code_snippet_manager_add_menu');
function code_snippet_manager_add_menu() {
    // Menú principal
    add_menu_page(
        __('Code Snippets Manager', 'code-snippet-manager'), 
        __('Code Snippets Manager', 'code-snippet-manager'), 
        'manage_options', 
        'code-snippet-manager', 
        'code_snippet_manager_main_page', 
        'dashicons-editor-code', 
        20
    );

    // Submenú CSS
    add_submenu_page(
        'code-snippet-manager', 
        __('CSS Snippets', 'code-snippet-manager'), 
        __('CSS Snippets', 'code-snippet-manager'), 
        'manage_options', 
        'edit.php?post_type=css_snippet'
    );
    add_submenu_page(
        'code-snippet-manager', 
        __('Add New CSS Snippet', 'code-snippet-manager'), 
        __('Add New CSS Snippet', 'code-snippet-manager'), 
        'manage_options', 
        'post-new.php?post_type=css_snippet'
    );

    // Submenú JS
    add_submenu_page(
        'code-snippet-manager', 
        __('JavaScript Snippets', 'code-snippet-manager'), 
        __('JavaScript Snippets', 'code-snippet-manager'), 
        'manage_options', 
        'edit.php?post_type=js_snippet'
    );
    add_submenu_page(
        'code-snippet-manager', 
        __('Add New JS Snippet', 'code-snippet-manager'), 
        __('Add New JS Snippet', 'code-snippet-manager'), 
        'manage_options', 
        'post-new.php?post_type=js_snippet'
    );

    // Submenú de configuración
    add_submenu_page(
        'code-snippet-manager', 
        __('Settings', 'code-snippet-manager'), 
        __('Settings', 'code-snippet-manager'), 
        'manage_options', 
        'code-snippet-manager-settings', 
        'code_snippet_manager_settings_page'
    );

    // Nueva subpágina para la vista de páginas e imágenes
    add_submenu_page(
        'code-snippet-manager', 
        __('Pages & Images', 'code-snippet-manager'), 
        __('Pages & Images', 'code-snippet-manager'), 
        'manage_options', 
        'code-snippet-manager-pages-images', 
        'code_snippet_manager_pages_images_page'
    );
}

// Callback para la página de páginas e imágenes
function code_snippet_manager_pages_images_page() {
    echo '<div class="wrap">';
    echo '<h1>' . __('Pages & Images Overview', 'code-snippet-manager') . '</h1>';
    echo do_shortcode('[contar_paginas_imagenes]');
    echo '</div>';
}


// Mantener el menú desplegado
add_filter('parent_file', 'code_snippet_manager_set_active_menu');
function code_snippet_manager_set_active_menu($parent_file) {
    global $submenu_file, $pagenow;

    // Revisar si estás editando o viendo un snippet de CSS
    if (get_post_type() === 'css_snippet') {
        $parent_file = 'code-snippet-manager'; // Menú principal activo
        if ($pagenow === 'post-new.php') {
            $submenu_file = 'post-new.php?post_type=css_snippet'; // Submenú activo
        } else {
            $submenu_file = 'edit.php?post_type=css_snippet'; // Submenú activo
        }
    }

    // Revisar si estás editando o viendo un snippet de JS
    if (get_post_type() === 'js_snippet') {
        $parent_file = 'code-snippet-manager'; // Menú principal activo
        if ($pagenow === 'post-new.php') {
            $submenu_file = 'post-new.php?post_type=js_snippet'; // Submenú activo
        } else {
            $submenu_file = 'edit.php?post_type=js_snippet'; // Submenú activo
        }
    }

    return $parent_file;
}


// Página principal del plugin
function code_snippet_manager_main_page() {
    echo '<div class="wrap">';
    echo '<h1>' . __('Welcome to Code Snippets Manager', 'code-snippet-manager') . '</h1>';
    echo '<p>' . __('This is the main page of the Code Snippet Manager plugin. From here, you can manage your CSS and JavaScript snippets.', 'code-snippet-manager') . '</p>';
    echo '<p><a href="' . admin_url('edit.php?post_type=css_snippet') . '">' . __('Manage CSS Snippets', 'code-snippet-manager') . '</a></p>';
    echo '<p><a href="' . admin_url('edit.php?post_type=js_snippet') . '">' . __('Manage JavaScript Snippets', 'code-snippet-manager') . '</a></p>';
    echo '</div>';
}

// Página de configuración del plugin
function code_snippet_manager_settings_page() {
    echo '<div class="wrap">';
    echo '<h1>' . __('Code Snippet Manager Settings', 'code-snippet-manager') . '</h1>';
    echo '<form method="post" action="options.php">';
    settings_fields('code_snippet_manager_settings'); // Este grupo debe coincidir con el del registro de ajustes
    do_settings_sections('code_snippet_manager'); // Este es el slug del formulario
    echo '<p><input type="submit" value="' . __('Save Settings', 'code-snippet-manager') . '" class="button-primary" /></p>';
    echo '</form>';
    echo '</div>';
}


// Ocultar el editor estándar
add_action('init', 'code_snippet_remove_editor_support');
function code_snippet_remove_editor_support() {
    remove_post_type_support('code_snippet', 'editor');
}

// Agregar un Meta Box para el editor de código
add_action('add_meta_boxes', 'code_snippet_add_meta_box');
function code_snippet_add_meta_box() {
    add_meta_box(
        'code_snippet_editor',
        __('Code Editor', 'code-snippet-manager'),
        'code_snippet_render_meta_box',
        ['css_snippet', 'js_snippet'],  // Asegura que el Meta Box aparezca tanto para CSS como para JS
        'normal',
        'high'
    );
}


/*------------------------------- Version Buena -------------------*/
function code_snippet_render_meta_box($post) {
    $post_type = get_post_type($post);
    
    // Obtener el contenido del post
    $post_content = get_post_field('post_content', $post->ID);
    $meta_data = json_decode($post_content, true);
    
    // Obtener el contenido guardado para cada tipo de fragmento
    $code_content = isset($meta_data['code_content']) ? $meta_data['code_content'] : get_post_meta($post->ID, '_code_snippet_content', true);

    // Si es un fragmento de CSS
    if ($post_type === 'css_snippet') {
        echo '<h2>' . __('CSS Code', 'code-snippet-manager') . '</h2>';
        echo '<textarea id="code_snippet_content" name="code_snippet_content" style="width:100%; height:300px; font-family: monospace;">' . esc_textarea($code_content) . '</textarea>';
    }
    // Si es un fragmento de JS
    if ($post_type === 'js_snippet') {
        echo '<h2>' . __('JavaScript Code', 'code-snippet-manager') . '</h2>';
        echo '<textarea id="code_snippet_content" name="code_snippet_content" style="width:100%; height:300px; font-family: monospace;">' . esc_textarea($code_content) . '</textarea>';
    }
}



add_action('save_post', 'code_snippet_save_meta_box_data');
function code_snippet_save_meta_box_data($post_id) {
    if (!current_user_can('edit_post', $post_id)) return;

    // Depuración para comprobar los metadatos
    if (isset($_POST['code_snippet_content'])) {
        error_log('Saving code_snippet_content: ' . $_POST['code_snippet_content']);
    }

    if (array_key_exists('code_snippet_content', $_POST)) {
        update_post_meta($post_id, '_code_snippet_content', $_POST['code_snippet_content']);
    }
}



// Agregar el Meta Box con opciones para dónde se debe aplicar el fragmento de código
add_action('add_meta_boxes', 'code_snippet_add_location_meta_box');
function code_snippet_add_location_meta_box() {
    add_meta_box(
        'code_snippet_location',
        __('Snippet Location Settings', 'code-snippet-manager'),
        'code_snippet_render_location_meta_box',
        ['css_snippet', 'js_snippet'], // Asegura que el Meta Box aparezca tanto para CSS como para JS
        'normal',
        'high'
    );
}











function code_snippet_render_location_meta_box($post) {
    // Obtener el contenido del post
    $post_content = get_post_field('post_content', $post->ID);
    $meta_data = json_decode($post_content, true);

    $incluidos = get_post_meta($post->ID, '_code_snippet_location_include', true);
    $excluidos = get_post_meta($post->ID, '_code_snippet_exclude_location', true);

    // Obtener configuraciones guardadas
    $location = isset($incluidos) ? $incluidos : [];
    $exclude_location = isset($excluidos) ? $excluidos : [];

    // Asegurarse de que sean arrays
    $location = is_array($location) ? $location : [];
    $exclude_location = is_array($exclude_location) ? $exclude_location : [];

    // Opciones generales
    $options_general = [
        'all' => __('All Site', 'code-snippet-manager'),
        'pages' => __('Pages Only', 'code-snippet-manager'),
        'home' => __('Home Page', 'code-snippet-manager'),
    ];
    

    // Ordenar las opciones generales por clave
    ksort($options_general);

    // Obtener todos los tipos de post públicos
    $post_types = get_post_types(['public' => true], 'objects');



    // Ordenar tipos de contenido por nombre singular
    uasort($post_types, function($a, $b) {
        return strcmp($a->labels->singular_name, $b->labels->singular_name);
    });

    // Selector de ubicación
    echo '<label for="code_snippet_location_include">' . __('Apply to:', 'code-snippet-manager') . '</label><br>';
    echo '<select name="code_snippet_location_include[]" id="code_snippet_location_include" multiple="multiple" style="width: 100%; height: 150px; resize: vertical; ">';

    // Opciones generales
    echo '<optgroup label="' . __('General Options', 'code-snippet-manager') . '">';
    foreach ($options_general as $key => $label) {
        echo '<option value="' . $key . '" ' . (in_array($key, $location) ? 'selected' : '') . '>' . $label . '</option>';
    }
    echo '</optgroup>';

    // Opciones por tipo de contenido y sus posts
    foreach ($post_types as $post_type) {

        if ($post_type->name === 'attachment') {
            continue;
        }

        echo '<optgroup label="' . sprintf(__('Type: %s', 'code-snippet-manager'), $post_type->labels->singular_name) . '">';

        // Opción para el tipo completo
        echo '<option value="post_type_' . $post_type->name . '" ' . (in_array('post_type_' . $post_type->name, $location) ? 'selected' : '') . '>';
        echo __('All ', 'code-snippet-manager') . $post_type->labels->name . '</option>';

        // Listar cada post del tipo
        $posts = get_posts([
            'post_type'      => $post_type->name,
            'posts_per_page' => -1,
            'post_status'    => ['publish', 'private', 'draft', 'pending']
        ]);

        // Ordenar publicaciones alfabéticamente por título
    usort($posts, function($a, $b) {
        // Normalizar títulos eliminando espacios y forzando minúsculas
        $titleA = strtolower(trim($a->post_title));
        $titleB = strtolower(trim($b->post_title));
        return strcmp($titleA, $titleB);
    });

        foreach ($posts as $single_post) {
            echo '<option value="post_' . $single_post->ID . '" ' . (in_array('post_' . $single_post->ID, $location) ? 'selected' : '') . '>';
            echo $single_post->post_title;
            echo '</option>';
        }

        echo '</optgroup>';
    }

    echo '</select>';

    // Selector de exclusión
    echo '<br><br><label for="code_snippet_exclude_location">' . __('Exclude from:', 'code-snippet-manager') . '</label><br>';
    echo '<select name="code_snippet_exclude_location[]" id="code_snippet_exclude_location" multiple="multiple" style="width: 100%; height: 150px; resize: vertical; ">';

    // Opción "None"
    echo '<optgroup label="' . __('General Options', 'code-snippet-manager') . '">';
    /* echo '<option value="" ' . (empty($exclude_location) ? 'selected' : '') . '> ' . __('None (Deseleccionar todo)', 'code-snippet-manager') . '</option>';*/
    foreach ($options_general as $key => $label) {
        echo '<option value="' . $key . '" ' . (in_array($key, $exclude_location) ? 'selected' : '') . '>' . $label . '</option>';
    }
    echo '</optgroup>';

    // Opciones por tipo de contenido y sus posts
    foreach ($post_types as $post_type) {
        if ($post_type->name === 'attachment') {
            continue;
        }

        echo '<optgroup label="' . sprintf(__('Type: %s', 'code-snippet-manager'), $post_type->labels->singular_name) . '">';

        // Opción para el tipo completo
        echo '<option value="post_type_' . $post_type->name . '" ' . (in_array('post_type_' . $post_type->name, $exclude_location) ? 'selected' : '') . '>';
        echo __('All ', 'code-snippet-manager') . $post_type->labels->name;
        echo '</option>';

        // Listar cada post del tipo
        $posts = get_posts([
            'post_type'      => $post_type->name,
            'posts_per_page' => -1,
            'post_status'    => ['publish', 'private', 'draft', 'pending']
        ]);

        // Ordenar publicaciones alfabéticamente por título
    usort($posts, function($a, $b) {
        // Normalizar títulos eliminando espacios y forzando minúsculas
        $titleA = strtolower(trim($a->post_title));
        $titleB = strtolower(trim($b->post_title));
        return strcmp($titleA, $titleB);
    });

        foreach ($posts as $single_post) {
            echo '<option value="post_' . $single_post->ID . '" ' . (in_array('post_' . $single_post->ID, $exclude_location) ? 'selected' : '') . '>';
            echo $single_post->post_title;
            echo '</option>';
        }

        echo '</optgroup>';
    }

    echo '</select>';
}



add_action('save_post', 'code_snippet_save_location_meta_box_data');
function code_snippet_save_location_meta_box_data($post_id) {
    // Verificar permisos de usuario
    if (!current_user_can('edit_post', $post_id)) return;

    // Guardar la ubicación de inclusión
    if (isset($_POST['code_snippet_location_include'])) {
        $location = $_POST['code_snippet_location_include'];
        if (!empty($location)) {
            update_post_meta($post_id, '_code_snippet_location_include', $location);
        } else {
            // Eliminar el metadato si está vacío
            delete_post_meta($post_id, '_code_snippet_location_include');
        }
    } else {
        // Eliminar el metadato si no existe en el POST
        delete_post_meta($post_id, '_code_snippet_location_include');
    }

    // Guardar la ubicación de exclusión
    if (isset($_POST['code_snippet_exclude_location'])) {
        $exclude_location = $_POST['code_snippet_exclude_location'];
        if (!empty($exclude_location)) {
            update_post_meta($post_id, '_code_snippet_exclude_location', $exclude_location);
        } else {
            // Eliminar el metadato si está vacío
            delete_post_meta($post_id, '_code_snippet_exclude_location');
        }
    } else {
        // Eliminar el metadato si no existe en el POST
        delete_post_meta($post_id, '_code_snippet_exclude_location');
    }
}






/* -------------- Metas Revisiones -------------- */

// Guardar los datos del Metabox en el contenido del post
add_action('save_post', 'code_snippet_save_meta_box_data_to_content');
function code_snippet_save_meta_box_data_to_content($post_id) {
    // Evitar ejecuciones innecesarias
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    // Verificar el tipo de post
    $post_type = get_post_type($post_id);
    if (!in_array($post_type, ['css_snippet', 'js_snippet'])) return;

    // Verificar si se está publicando o actualizando
    if (isset($_POST['action']) && ($_POST['action'] === 'publish' || $_POST['action'] === 'editpost')) {
        // Inicializar una variable para el contenido plano
        $plain_text_content = '';

        // Validar y agregar el contenido del snippet
        if (isset($_POST['code_snippet_content'])) {
            // Sanitizar el contenido del código
            $plain_text_content .= "Snippet Content:\n" . wp_kses_post($_POST['code_snippet_content']) . "\n\n";
        }

        // Validar y agregar las ubicaciones de inclusión
        if (isset($_POST['code_snippet_location_include']) && is_array($_POST['code_snippet_location_include'])) {
            $locations = implode(', ', array_map('sanitize_text_field', $_POST['code_snippet_location_include']));
            $plain_text_content .= "Include Locations:\n" . $locations . "\n\n";
        }

        // Validar y agregar las ubicaciones de exclusión
        if (isset($_POST['code_snippet_exclude_location']) && is_array($_POST['code_snippet_exclude_location'])) {
            $exclude_locations = implode(', ', array_map('sanitize_text_field', $_POST['code_snippet_exclude_location']));
            $plain_text_content .= "Exclude Locations:\n" . $exclude_locations . "\n\n";
        }

        // Guardar el texto plano en el contenido del post
        remove_action('save_post', 'code_snippet_save_meta_box_data_to_content'); // Evitar bucles
        wp_update_post([
            'ID'           => $post_id,
            'post_content' => $plain_text_content, // Guardar el texto plano
        ]);
        add_action('save_post', 'code_snippet_save_meta_box_data_to_content'); // Reagregar la acción
    }
}





// Mostrar el contenido del JSON en el editor de WordPress
add_filter('the_content', 'code_snippet_render_content_from_meta', 10, 1);
function code_snippet_render_content_from_meta($content) {
    if (is_singular(['css_snippet', 'js_snippet']) && in_the_loop() && is_main_query()) {
        $meta_data = json_decode(get_post_field('post_content', get_the_ID()), true);

        if ($meta_data) {
            $output = '';
            // Mostrar el código del snippet
            if (isset($meta_data['code_content'])) {
                $output .= '<h2>' . __('Code Content', 'code-snippet-manager') . '</h2>';
                $output .= '<pre>' . esc_html($meta_data['code_content']) . '</pre>';
            }
            // Mostrar las ubicaciones de inclusión
            if (isset($meta_data['location'])) {
                $output .= '<h3>' . __('Locations', 'code-snippet-manager') . '</h3>';
                $output .= '<ul>';
                foreach ($meta_data['location'] as $location) {
                    $output .= '<li>' . esc_html($location) . '</li>';
                }
                $output .= '</ul>';
            }
            // Mostrar las ubicaciones de exclusión
            if (isset($meta_data['exclude_location'])) {
                $output .= '<h3>' . __('Exclude Locations', 'code-snippet-manager') . '</h3>';
                $output .= '<ul>';
                foreach ($meta_data['exclude_location'] as $exclude) {
                    $output .= '<li>' . esc_html($exclude) . '</li>';
                }
                $output .= '</ul>';
            }
            return $content . $output;
        }
    }
    return $content;
}


function ocultar_boton_restaurar_versiones() {
    // Verificar si estamos en una página de revisión
    if (isset($_GET['revision']) && is_admin()) {
        // Obtener el ID de la revisión
        $revision_id = $_GET['revision'];
        
        // Obtener la revisión utilizando wp_get_post_revision
        $revision = wp_get_post_revision($revision_id);
        
        // Si no hay revisión, salir
        if (!$revision) return;
        
        // Obtener el post original (post asociado a la revisión)
        $post = get_post($revision->post_parent);
        
        // Verificar si el tipo de post es 'css_snippet' o 'js_snippet'
        if ($post && ($post->post_type === 'css_snippet' || $post->post_type === 'js_snippet')) {
            // Inyectar el estilo para ocultar el botón de restaurar versión
            echo '<style>
                .restore-revision {
                    display: none !important;
                }

                .diff-title{
                display:flex;
                }
            </style>';

            add_action('admin_notices', function() {
                echo '<div class="notice notice-warning is-dismissible">
                        <p><strong>Aviso:</strong> No es posible restaurar versiones de fragmentos de código CSS/JS. Esta herramienta solo permite obtener variables previas sin la opción de reversión.</p>
                    </div>';
            });

        }
        
    }
}
add_action('admin_head', 'ocultar_boton_restaurar_versiones');




// Agregar un Meta Box para seleccionar si el código se aplica al header o footer
add_action('add_meta_boxes', 'code_snippet_add_position_meta_box');
function code_snippet_add_position_meta_box() {
    add_meta_box(
        'code_snippet_position',
        __('Apply Code To', 'code-snippet-manager'),
        'code_snippet_render_position_meta_box',
        ['js_snippet'], // se puede tambien agregar css_snippet
        'side',
        'default'
    );
}

// Renderizar el Meta Box
function code_snippet_render_position_meta_box($post) {
    // Obtener la configuración guardada o establecer 'header' como predeterminado
    $snippet_position = get_post_meta($post->ID, '_code_snippet_position', true);
    if (empty($snippet_position)) {
        $snippet_position = 'header'; // Valor por defecto
    }

    // Opciones de posición
    $options = [
        'header' => __('Header', 'code-snippet-manager'),
        'footer' => __('Footer', 'code-snippet-manager'),
    ];

    // Renderizar opciones
    echo '<fieldset>';
    foreach ($options as $value => $label) {
        echo '<label>';
        echo '<input type="radio" name="code_snippet_position" value="' . esc_attr($value) . '" ' . checked($snippet_position, $value, false) . '> ' . esc_html($label);
        echo '</label><br>';
    }
    echo '</fieldset>';
}

// Guardar la posición seleccionada
add_action('save_post', 'code_snippet_save_position_meta_box_data');
function code_snippet_save_position_meta_box_data($post_id) {
    if (!current_user_can('edit_post', $post_id)) return;

    if (isset($_POST['code_snippet_position'])) {
        update_post_meta($post_id, '_code_snippet_position', sanitize_text_field($_POST['code_snippet_position']));
    }
}





// Agregar columnas personalizadas a la lista de posts
add_filter('manage_css_snippet_posts_columns', 'add_code_snippet_columns');
add_filter('manage_js_snippet_posts_columns', 'add_code_snippet_columns');
function add_code_snippet_columns($columns) {
    $columns['included_pages'] = __('Included Pages', 'code-snippet-manager');
    $columns['excluded_pages'] = __('Excluded Pages', 'code-snippet-manager');
    return $columns;
}

// Mostrar los datos de las columnas personalizadas
add_action('manage_css_snippet_posts_custom_column', 'display_code_snippet_columns', 10, 2);
add_action('manage_js_snippet_posts_custom_column', 'display_code_snippet_columns', 10, 2);
function display_code_snippet_columns($column_name, $post_id) {
    if ($column_name == 'included_pages') {
        // Obtener las páginas incluidas
        $included_pages = get_post_meta($post_id, '_code_snippet_location_include', true);
        if ($included_pages) {
            // Mostrar las opciones seleccionadas como texto con estilos
            $included_text = [];
            foreach ($included_pages as $page_id) {
                if ($page_id === 'all') {
                    $included_text[] = '<span class="included-option_cs all-site">' . __('All Site', 'code-snippet-manager') . '</span>';
                } elseif ($page_id === 'pages') {
                    $included_text[] = '<span class="included-option_cs pages-only">' . __('Pages Only', 'code-snippet-manager') . '</span>';
                } elseif ($page_id === 'home') {
                    $home_url = home_url();
                    $included_text[] = '<a href="' . esc_url($home_url) . '" target="_blank"><span class="included-option_cs home-page">' . __('Home Page', 'code-snippet-manager') . '</span></a>';
                } elseif (strpos($page_id, 'post_type_') === 0) {
                    $post_type_name = str_replace('post_type_', '', $page_id);
                    $post_type = get_post_type_object($post_type_name);
                    $url = get_post_type_archive_link($post_type_name);
                    $included_text[] = '<a href="' . esc_url($url) . '" target="_blank"><span class="included-option_cs post-type">' . __('All', 'code-snippet-manager') . ' ' . $post_type->labels->name . '</span></a>';
                } elseif (strpos($page_id, 'post_') === 0) {
                    $post_id = str_replace('post_', '', $page_id);
                    $post = get_post($post_id);
                    if ($post) {
                        $url = get_permalink($post_id);
                        $included_text[] = '<a href="' . esc_url($url) . '" target="_blank"><span class="included-option_cs post-title">' . $post->post_title . '</span></a>';
                    }
                }
            }
            echo implode(' ', $included_text); // Mostrar los textos con las etiquetas span sin comas
        } else {
            echo '<span class="included-option_cs none">' . __('None', 'code-snippet-manager') . '</span>';
        }
    }

    if ($column_name == 'excluded_pages') {
        // Obtener las páginas excluidas
        $excluded_pages = get_post_meta($post_id, '_code_snippet_exclude_location', true);
        if ($excluded_pages) {
            // Mostrar las opciones excluidas como texto con estilos
            $excluded_text = [];
            foreach ($excluded_pages as $page_id) {
                if ($page_id === 'all') {
                    $excluded_text[] = '<span class="excluded-option_cs all-site">' . __('All Site', 'code-snippet-manager') . '</span>';
                } elseif ($page_id === 'pages') {
                    $excluded_text[] = '<span class="excluded-option_cs pages-only">' . __('Pages Only', 'code-snippet-manager') . '</span>';
                } elseif ($page_id === 'home') {
                    $home_url = home_url();
                    $excluded_text[] = '<a href="' . esc_url($home_url) . '" target="_blank"><span class="excluded-option_cs home-page">' . __('Home Page', 'code-snippet-manager') . '</span></a>';
                } elseif (strpos($page_id, 'post_type_') === 0) {
                    $post_type_name = str_replace('post_type_', '', $page_id);
                    $post_type = get_post_type_object($post_type_name);
                    $url = get_post_type_archive_link($post_type_name);
                    $excluded_text[] = '<a href="' . esc_url($url) . '" target="_blank"><span class="excluded-option_cs post-type">' . __('All', 'code-snippet-manager') . ' ' . $post_type->labels->name . '</span></a>';
                } elseif (strpos($page_id, 'post_') === 0) {
                    $post_id = str_replace('post_', '', $page_id);
                    $post = get_post($post_id);
                    if ($post) {
                        $url = get_permalink($post_id);
                        $excluded_text[] = '<a href="' . esc_url($url) . '" target="_blank"><span class="excluded-option_cs post-title">' . $post->post_title . '</span></a>';
                    }
                }
            }
            echo implode(' ', $excluded_text); // Mostrar los textos con las etiquetas span sin comas
        } else {
            echo '<span class="excluded-option_cs none">' . __('None', 'code-snippet-manager') . '</span>';
        }
    }
}





function mostrar_aviso_edicion_pagina_actual($wp_admin_bar) {
    if (!is_admin() && is_user_logged_in() && is_singular()) {
        global $wpdb, $post;

        $post_id = get_queried_object_id();
        $mostrar_aviso = false;
        $username = 'Usuario desconocido';

        // 🟢 1. Verificar si alguien está editando en el editor clásico
        $edit_lock = get_post_meta($post_id, '_edit_lock', true);
        if ($edit_lock) {
            list($timestamp, $user_id) = explode(":", $edit_lock);
            if ((time() - $timestamp) < 150 && $user_id) {
                $user_info = get_userdata($user_id);
                if ($user_info) {
                    $username = $user_info->display_name;
                    $mostrar_aviso = true;
                }
            }
        }

        // 🟢 2. Detectar si Elementor está en edición en este momento
        if (isset($_GET['action']) && $_GET['action'] === 'elementor') {
            $current_user = wp_get_current_user();
            if ($current_user) {
                $username = $current_user->display_name;
                $mostrar_aviso = true;
            }
        }

        // 🟢 3. Detectar si WPBakery está activo en frontend
        if (isset($_GET['vc_editable']) && $_GET['vc_editable'] === 'true' || isset($_COOKIE['vc_post_settings'])) {
            // Verificar si alguien está editando la página con WPBakery
            $user_info = get_userdata(get_current_user_id());
            if ($user_info) {
                $username = $user_info->display_name;
                $mostrar_aviso = true;

                // Establecer un "lock" temporal en los metadatos para esta página
                update_post_meta($post_id, '_edit_lock', time() . ":" . get_current_user_id());
            }
        }

        // 🟢 Mostrar aviso en la barra de admin si hay edición activa
        if ($mostrar_aviso) {
            $mensaje = "⚠️ $username está editando";
            $wp_admin_bar->add_node([
                'id'    => 'aviso_edicion_actual',
                'title' => $mensaje,
                /* 'href'  => get_edit_post_link($post_id), */
                'meta'  => ['class' => 'aviso-edicion']
            ]);
        }
    }
}
add_action('admin_bar_menu', 'mostrar_aviso_edicion_pagina_actual', 100);






// Encolar el script JS solo cuando WPBakery está activo en el frontend
function agregar_script_actualizar_lock() {
    // Verificar si estamos en una página singular y si WPBakery está activo
    if (is_singular() && is_user_logged_in() && (isset($_GET['vc_editable']) && $_GET['vc_editable'] === 'true' || isset($_COOKIE['vc_post_settings']))) {
        ?>
        <script type="text/javascript">
            // Actualizar el lock de edición cada 30 segundos
            setInterval(function() {
                var post_id = <?php echo get_queried_object_id(); ?>;
                var data = {
                    action: 'actualizar_lock_edicion',
                    post_id: post_id
                };

                // Enviar solicitud AJAX para actualizar el lock
                jQuery.post(ajaxurl, data, function(response) {
                    // Respuesta (si es necesario)
                    console.log('Lock actualizado');
                });
            }, 30000); // 30 segundos
        </script>
        <?php
    }
}
add_action('wp_footer', 'agregar_script_actualizar_lock');



// Manejo de la solicitud AJAX para actualizar el lock
function actualizar_lock_edicion_ajax() {
    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    if ($post_id) {
        // Actualizar el lock en los metadatos
        update_post_meta($post_id, '_edit_lock', time() . ":" . get_current_user_id());
        wp_send_json_success();
    }
    wp_send_json_error();
}
add_action('wp_ajax_actualizar_lock_edicion', 'actualizar_lock_edicion_ajax');




//Meta boxes para atributos 


function code_snippet_add_attributes_metabox() {
    $post_types = ['css_snippet', 'js_snippet'];

    foreach ($post_types as $post_type) {
        add_meta_box(
            'code_snippet_attributes',
            'Atributos adicionales para la etiqueta',
            'code_snippet_attributes_metabox_callback',
            $post_type,
            'normal',
            'default'
        );
    }
}
add_action('add_meta_boxes', 'code_snippet_add_attributes_metabox');

function code_snippet_attributes_metabox_callback($post) {
    $attributes = get_post_meta($post->ID, '_code_snippet_attributes', true);
    echo '<textarea style="width:100%" rows="3" name="code_snippet_attributes">' . esc_textarea($attributes) . '</textarea>';
    echo '<p class="description">Agrega atributos personalizados para la etiqueta &lt;script&gt; o &lt;style&gt;. Ejemplo: <code>defer data-name="demo"</code></p>';
}

function code_snippet_save_attributes_metabox($post_id) {
    if (isset($_POST['code_snippet_attributes'])) {
        update_post_meta($post_id, '_code_snippet_attributes', sanitize_text_field($_POST['code_snippet_attributes']));
    }
}
add_action('save_post', 'code_snippet_save_attributes_metabox');


